( function ( $ ) {

    var tc_braintree3ds2 = {
        params : JSON.parse( braintree_params ),
        dropin: '',
        amount: '',
        clientToken: '',
        ajaxUrl: '',
        container: $( '#braintree_3ds2.tc_gateway_form' ),
        errorContainer: document.getElementById( 'errors' ),
        payBtn: document.getElementById( 'pay-btn-3ds2' ),
        methodBraintree: document.getElementById( 'braintree_3ds2' ),
        tableBraintree: document.getElementById( 'tbl_braintree' ),
        overlay: document.getElementById( 'braintree_overlay' ),
        paymentForm: document.getElementById( 'tc_payment_form' ),
        get paymentFormChildCount() {
            return this.paymentForm.childElementCount
        },
        nonceGroup: document.querySelector( '.nonce-group' ),
        nonceInput: document.querySelector( '.nonce-group input' ),
        nonceSpan: document.querySelector( '.nonce-group span' ),
        payGroup: document.querySelector( '.pay-group' ),
        billingFields: [
            'billing-email',
            'billing-phone',
            'billing-first-name',
            'billing-last-name',
            'billing-street-address',
            'billing-extended-address',
            'billing-city',
            'billing-region',
            'billing-postal-code',
            'billing-country-code'
        ].reduce( function( fields, fieldName ) {
            var field = fields[ fieldName ] = {
                input: document.getElementById( fieldName ),
                help: document.getElementById( 'help-' + fieldName )
            };

            field.input.addEventListener( 'focus', function() {
                clearFieldValidations( field );
            } );

            return fields;
        }, {} )
    };

    tc_braintree3ds2.billingFields[ 'billing-extended-address' ].optional = true;

    /**
     * Remove Error messages
     * @param field
     */
    function clearFieldValidations( field ) {
        field.help.innerText = '';
        field.help.parentNode.classList.remove( 'has-error' );
    }

    /**
     * Validate Billing Fields
     * @returns {boolean}
     */
    function validateBillingFields() {

        var isValid = true;

        Object.keys( tc_braintree3ds2.billingFields ).forEach( function( fieldName ) {

            var fieldEmpty = false,
                field = tc_braintree3ds2.billingFields[ fieldName ];

            if ( field.optional ) {
                return;
            }

            fieldEmpty = field.input.value.trim() === '';

            if ( fieldEmpty ) {
                isValid = false;
                field.help.innerText = tc_braintree3ds2.params.billing_error;
                field.help.parentNode.classList.add( 'has-error' );

            } else {
                clearFieldValidations( field );
            }
        } );

        return isValid;
    }

    /**
     * Initialize Braintree Dropin connection
     */
    function start() {

        // Assign global amount value for threeDSecure
        tc_braintree3ds2.amount = tc_braintree3ds2.params.amount;

        // Assign global ajaxUrl value for payment_method_nonce
        tc_braintree3ds2.ajaxUrl = tc_braintree3ds2.params.callback;

        // Pass Client Token for Verification
        tc_braintree3ds2.clientToken = tc_braintree3ds2.params.token;

        onFetchClientToken( tc_braintree3ds2.clientToken );
    }

    /**
     * Initialize Braintree Dropin form
     * @param clientToken
     * @returns {*}
     */
    function onFetchClientToken( clientToken ) {
        return setupDropin( clientToken ).then( function( instance ) {
            tc_braintree3ds2.dropin = instance;
            tc_braintree3ds2.payBtn.style.display = 'block';
            setupForm();

        } ).catch( function( err ) {
            onFetchResponseError( err );
        } );
    }

    /**
     * Display error messages
     * @param err
     */
    function onFetchResponseError( err ) {
        tc_braintree3ds2.tableBraintree.style.display = 'none';
        tc_braintree3ds2.errorContainer.innerText = "Error: " + err.message;
    }

    /**
     * Create Braintree Dropin form
     * @param clientToken
     * @returns {*}
     */
    function setupDropin( clientToken ) {
        return braintree.dropin.create( {
            authorization: clientToken,
            container: '#drop-in',
            threeDSecure: true,
        } )
    }

    /**
     * Enable Pay button on Braintree Initialization
     */
    function setupForm() {
        enablePayNow();
    }

    /**
     * Enable/Display Pay button
     */
    function enablePayNow() {

        tc_braintree3ds2.payBtn.value = tc_braintree3ds2.params.pay_now;
        tc_braintree3ds2.payBtn.removeAttribute( 'disabled' );

        // Fixing max-height for accordion effect.
        tc_braintree3ds2.container.css( 'max-height', '100%' );

        // Delay a bit to capture the final container height
        setTimeout( function() {
            tc_braintree3ds2.container.css( 'max-height', tc_braintree3ds2.container.height() + 'px' )
        }, 100 )
    }

    /**
     * Display payload nonce
     * @param payload
     * @param liabilityShift
     */
    function showNonce( payload, liabilityShift ) {
        tc_braintree3ds2.nonceSpan.textContent = tc_braintree3ds2.params.liability_shifted + liabilityShift;
        tc_braintree3ds2.nonceInput.value = payload.nonce;
        tc_braintree3ds2.payGroup.classList.add( 'hidden' );
        tc_braintree3ds2.payGroup.style.display = 'none';
        tc_braintree3ds2.nonceGroup.classList.remove( 'hidden' );
    }

    /**
     * Process payment and create Tickera order
     * @param nonce
     */
    function nonceCallback( nonce ) {

        var xhttp = new XMLHttpRequest(),
            params = 'nonce=' + nonce;

        xhttp.open( 'POST', tc_braintree3ds2.ajaxUrl, true );

        // Send the proper header information along with the request
        xhttp.setRequestHeader( 'Content-type', 'application/x-www-form-urlencoded' );
        xhttp.onreadystatechange = function() {//Call a function when the state changes.
            if ( xhttp.readyState == 4 && xhttp.status == 200 ) {
                let response = JSON.parse( xhttp.responseText );
                if ( response ) {
                    window.location.assign( response );
                } else {
                    window.location.reload();
                }
            }
        }

        xhttp.send( params );
    }

    /**
     * Validate if Numeric value
     * @param evt
     * @returns {boolean}
     */
    function isNumeric( evt ) {
        let charCode = ( evt.which ) ? evt.which : event.keyCode
        if ( charCode > 31 && ( charCode < 48 || charCode > 57 ) )
            return false;
        return true;
    }

    /**
     * Prepare error messages
     * @param err
     * @param ref
     */
    function errorMessage( err, ref = null ) {
        tc_braintree3ds2.errorContainer.innerText = err;

        if ( ref != null ) {
            let anchor = document.createElement( 'a' );
            anchor.setAttribute( 'href', ref );
            anchor.setAttribute( 'target', 'blank' );
            anchor.innerText = 'Three D Secure Validation Errors';
            tc_braintree3ds2.errorContainer.append( anchor );
        }

        document.getElementById( 'tc_payment_form' ).scrollIntoView( { block: 'center', behavior: 'smooth' } );
    }

    /**
     * Request payment method to Braintree Gateway and validation
     */
    tc_braintree3ds2.payBtn.addEventListener( 'click', function( event ) {

        event.preventDefault();
        tc_braintree3ds2.payBtn.setAttribute( 'disabled', 'disabled' );
        tc_braintree3ds2.payBtn.value = tc_braintree3ds2.params.processing;

        // Validate input fields
        var billingIsValid = validateBillingFields();
        if ( !billingIsValid ) {
            enablePayNow();
            return;
        }

        tc_braintree3ds2.dropin.requestPaymentMethod( {
            threeDSecure: {
                amount: tc_braintree3ds2.amount,
                email: tc_braintree3ds2.billingFields[ 'billing-email' ].input.value,
                billingAddress: {
                    givenName: tc_braintree3ds2.billingFields[ 'billing-first-name' ].input.value,
                    surname: tc_braintree3ds2.billingFields[ 'billing-last-name' ].input.value,
                    phoneNumber: tc_braintree3ds2.billingFields[ 'billing-phone' ].input.value.replace( /[\(\)\s\-]/g, '' ), // remove (), spaces, and - from phone number
                    streetAddress: tc_braintree3ds2.billingFields[ 'billing-street-address' ].input.value,
                    extendedAddress: tc_braintree3ds2.billingFields[ 'billing-extended-address' ].input.value,
                    locality: tc_braintree3ds2.billingFields[ 'billing-city' ].input.value,
                    postalCode: tc_braintree3ds2.billingFields[ 'billing-postal-code' ].input.value,
                    countryCodeAlpha3: tc_braintree3ds2.billingFields[ 'billing-country-code' ].input.value,
                    // region: billingFields['billing-region'].input.value, // TODO: Braintee unable to detect country code when region is enabled
                }
            }
        }, function( err, payload ) {

            if ( err ) {

                tc_braintree3ds2.dropin.clearSelectedPaymentMethod();

                let message = tc_braintree3ds2.params.tokenization_error,
                    refLink = 'https://developers.braintreepayments.com/reference/general/validation-errors/all/php',
                    errOriginal = err.details.originalError;

                if ( typeof errOriginal.details != 'undefined' ) {

                    message = message +
                        err.message + '\n\r' +
                        errOriginal.details.originalError.error.message + '\n\r';

                    errorMessage( message, refLink );

                } else {
                    errorMessage( message + err.message + '\n\r', refLink );
                }

                enablePayNow();
                return;
            }

            if ( !payload.liabilityShifted ) {
                tc_braintree3ds2.dropin.clearSelectedPaymentMethod();
                errorMessage( tc_braintree3ds2.params.process_error );
                enablePayNow();
                return;
            }

            tc_braintree3ds2.overlay.style.display = 'block';
            nonceCallback( payload.nonce );
        } );
    } );

    if (
        tc_braintree3ds2.paymentFormChildCount > 1
        && ! $( '#braintree_3ds2' ).closest( '.tickera-payment-gateways' ).hasClass( 'active' ) )
    {
        tc_braintree3ds2.methodBraintree.addEventListener( 'click', function( event ) {
            let dropIn = document.getElementById( 'drop-in' );
            if ( !dropIn.children.length ) {
                start();
            }
        } );

    } else {
        let dropIn = document.getElementById( 'drop-in' );
        if ( !dropIn.children.length ) {
            start();
        }
    }

    // Select the node that will be observed for mutations
    const dropInNode = document.getElementById( 'drop-in' );

    // Options for the observer (which mutations to observe)
    const dropInConfig = { childList: true };

    // Callback function to execute when mutations are observed
    const dropInCallback = function( mutationsList, dropInObserver ) {
        // Use traditional 'for loops' for IE 11
        for ( let mutation of mutationsList ) {
            if ( mutation.type === 'childList' ) {
                tc_braintree3ds2.tableBraintree.style.display = 'table';
            }
        }

        // Stop observing
        dropInObserver.disconnect();
    };

    // Create an observer instance linked to the callback function
    const dropInObserver = new MutationObserver( dropInCallback );

    // Start observing the target node for configured mutations
    dropInObserver.observe( dropInNode, dropInConfig );

    /**
     *  Initialize Country and Region Fields
     */
    for ( let i = 0; i < tc_braintree3ds2.params.country_data.length; i++ ) {
        $( '#tbl_braintree #billing-country-code' ).append( '<option value="' + tc_braintree3ds2.params.country_data[ i ].id + '">' + tc_braintree3ds2.params.country_data[ i ].text + '</option>' );
    }

    /**
     * Update regions based on selected country
     */
    $( document ).on( 'change', '#tbl_braintree #billing-country-code', function() {

        let selected_country = $( this ).val();

        // Make sure to empty field before process
        $( '#tbl_braintree #billing-region' ).empty();
        $( '#tbl_braintree #billing-region' ).attr( 'disabled', true );

        $( tc_braintree3ds2.params.region_data ).each( function( index, elem ) {
            if ( elem.countryShortCode == selected_country ) {
                for ( let i = 0; i < elem.regions.length; i++ ) {
                    $( '#tbl_braintree #billing-region' ).append( '<option value="' + elem.regions[ i ].name + '">' + elem.regions[ i ].name + ' | ' + elem.regions[ i ].shortCode + '</option>' );
                }
                $( '#tbl_braintree #billing-region' ).attr( 'disabled', false );
            }
        } );
    } );
})( jQuery );
